(***************************************************
TColorPalette->TComponent

Manages a 256 color logical palette.

Properties

BadEntries-
  If this palette is not an identity palette, this StringList
  will contain the palette entries that are responsible and
  require alteration.
IdentityPalette-
  Indicates whether this palette qualifies as an identity palette.
ImagePalette-
  Assigning this property to a TImage on the form causes the
  palette of the TImage to be copied into the TColorPalette.
  The reference to the TImage is not maintained, but the physical
  palette data is.
PalEntryFlag-
  Indicates if the palette entries should be flagged as PC_NOCOLLAPSE
  or PC_RESERVED.  The recommended setting is PC_NOCOLLPASE.
Palette-
  The HPalette handle that corresponds to the logical palette.
PaletteEntry[n]-
  Accesses the palette entry structure at the specified index.
  When modifying palette entries, call the Refresh method
  to regenerate the palette.
PaletteEntries-
  In design mode brings up a property editor that allows you
  to manipulate the palette visually.

Events

Methods

Refresh-
  Causes the logical palette to regenerate.  Call this after changing
  PaletteEntry values.
***************************************************)

{$R-}
unit ColorPalette;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, Utility, DsgnIntf;

type

  EPalette = class( Exception );

  TPAL = record
     logPalette: TLogPalette;
     palpalEntry: array[0..255] of TPaletteEntry;
  end;

  TPalEntryFlag = ( pcNoCollapse, pcReserved );

  TColorPalette = class( TComponent )
  private
     bEntries: boolean;
     FEntries: TStrings;
     FFlag: TPalEntryFlag;
     FFlag_: byte;
     FPalette: HPALETTE;
     FImage: TImage;
     bDummy: boolean;
     nDummy: integer;
     FBadEntries: TStrings;
     bAnimating: boolean;
     procedure PaletteEntriesToStrings;
     procedure SetPaletteHandle( bStoreStrings: boolean );
  protected
     procedure Loaded; override;
     procedure SetFlag( f: TPalEntryFlag );
     procedure SetImage( im: TImage );
     procedure SetEntries( str: TStrings );
     function IsIdentityPalette: boolean;
     procedure SetBadEntry( str: TStrings );
     function GetPaletteEntry( n: byte ): TPaletteEntry;
     procedure SetPaletteEntry( n: byte; pal: TPaletteEntry );
  public
     pal: TPAL;
     constructor Create( AOwner: TComponent ); override;
     destructor Destroy; override;
     procedure Assign( Source: TPersistent ); override;
     procedure Refresh;
     procedure beginAnimation;
     procedure endAnimation( StartIndex, NbrOfEntries: integer );
     property Palette: HPALETTE read FPalette write FPalette;
     property PaletteEntry[n: byte]: TPaletteEntry read GetPaletteEntry write SetPaletteEntry;
  published
     property BadEntries: TStrings read FBadEntries write SetBadEntry;
     property PaletteEntries: TStrings read FEntries write SetEntries;
     property ImagePalette: TImage read FImage write SetImage;
     property IdentityPalette: boolean read IsIdentityPalette write bDummy;
     property PalEntryFlag: TPalEntryFlag read FFlag write SetFlag default pcNoCollapse;
  end;

  TColorPaletteEditor = class( TPropertyEditor )
  private
  protected
  public
     procedure Edit; override;
     function GetAttributes: TPropertyAttributes; override;
     function GetValue: string; override;
  end;

procedure Register;

implementation

uses
  ColorPaletteEditor;

constructor TColorPalette.Create( AOwner: TComponent );
var
  hScreenDC: HDC;
  i: integer;
begin
  inherited Create( AOwner );

  bEntries := FALSE;
  FEntries := TStringList.Create;
  FBadEntries := TStringList.Create;

  FFlag := pcNoCollapse;
  FFlag_ := PC_NOCOLLAPSE;

  pal.logPalette.palVersion := $300;
  pal.logPalette.palNumEntries := 256;

{ Get the screen DC }
  hScreenDC := GetDC( 0 );

{ Copy 20 standard system colors }
  GetSystemPaletteEntries( hScreenDC, 0, 10, pal.logPalette.palpalEntry[0] );
  i := 246;
  GetSystemPaletteEntries( hScreenDC, 246, 10, pal.logPalette.palpalEntry[i] );

{ Get rid of screen DC }
  ReleaseDC( 0, hScreenDC );

{ Fill the rest of the entries with grayscale }
  for i := 10 to 245 do
     begin
        pal.logPalette.palpalEntry[i].peRed := i;
        pal.logPalette.palpalEntry[i].peGreen := i;
        pal.logPalette.palpalEntry[i].peBlue := i;
        pal.logPalette.palpalEntry[i].peFlags := FFlag_;
     end;

  PaletteEntriesToStrings;

end;

destructor TColorPalette.Destroy;
begin
  FEntries.Free;
  FBadEntries.Free;
  if FPalette <> 0 then
     DeleteObject( FPalette );
  inherited Destroy;
end;

procedure TColorPalette.SetFlag( f: TPalEntryFlag );
begin
  FFlag := f;
  if f = pcNoCollapse then
     FFlag_ := PC_NOCOLLAPSE
  else
     FFlag_ := PC_RESERVED;
end;

procedure TColorPalette.SetPaletteHandle( bStoreStrings: boolean );
begin
  if FPalette <> 0 then
     DeleteObject( FPalette );

  if ( FEntries.Count > 0 ) and not bEntries then
     SetEntries( FEntries );

  FPalette := CreatePalette( pal.logPalette );
  if FPalette = 0 then
     raise EPalette.Create( 'CreatePalette failed with Error Code: ' + IntToStr( GetLastError ) );

  if bStoreStrings then
     PaletteEntriesToStrings;
end;

(***************************************************
Check the middle 236 colors against the first 10
system colors.
***************************************************)
function TColorPalette.IsIdentityPalette: boolean;
var
  i, k: integer;
begin
  Result := TRUE;
  FBadEntries.Clear;
  for i := 10 to 245 do
     for k := 0 to 9 do
        begin
           if pal.logPalette.palpalEntry[i].peRed = pal.logPalette.palpalEntry[k].peRed then
              if pal.logPalette.palpalEntry[i].peGreen = pal.logPalette.palpalEntry[k].peGreen then
                 if pal.logPalette.palpalEntry[i].peBlue = pal.logPalette.palpalEntry[k].peBlue then
                    begin
                       FBadEntries.Add( IntToStr( i ) );
                       Result := FALSE;
                    end;
        end;
end;

(***************************************************
Set the palette entries based on the TImage component
***************************************************)
procedure TColorPalette.SetImage( im: TImage );
var
  palTemp: array[0..255] of TPaletteEntry;
  i: integer;
begin
  GetPaletteEntries( im.Picture.Bitmap.Palette, 0, 256, palTemp[0] );
  for i := 10 to 245 do
     begin
        pal.logPalette.palpalEntry[i].peRed := palTemp[i].peRed;
        pal.logPalette.palpalEntry[i].peGreen := palTemp[i].peGreen;
        pal.logPalette.palpalEntry[i].peBlue := palTemp[i].peBlue;
        pal.logPalette.palpalEntry[i].peFlags := FFlag_;
     end;
  SetPaletteHandle( TRUE );
  if csDesigning in ComponentState then
     ShowMessage( 'Palette Data Loaded from TImage ' + im.Name );
end;

(***************************************************
The user has modified the string list!
***************************************************)
procedure TColorPalette.SetEntries( str: TStrings );
var
  i: integer;
  sEntry: string;
begin
  bEntries := TRUE;
  if str.Count > 0 then
     begin
        if str.Count <> 256 then
           raise EPalette.Create( 'Exactly 256 entries must exist' );

        if str <> FEntries then
           FEntries.Assign( str );

        for i := 0 to 255 do
           begin
              sEntry := FEntries[i];
              pal.logPalette.palpalEntry[i].peRed := StrToInt( GetToken( sEntry, ',' ) );
              pal.logPalette.palpalEntry[i].peGreen := StrToInt( GetToken( sEntry, ',' ) );
              pal.logPalette.palpalEntry[i].peBlue := StrToInt( GetToken( sEntry, ',' ) );
              pal.logPalette.palpalEntry[i].peFlags := FFlag_;
           end;

        SetPaletteHandle( FALSE );
     end;
end;

procedure TColorPalette.Loaded;
begin
  inherited Loaded;
  if FEntries.Count > 0 then
     SetEntries( FEntries );
  SetPaletteHandle( TRUE );
end;

procedure TColorPalette.PaletteEntriesToStrings;
var
  i: integer;
  sEntry: string;
begin
  FEntries.Clear;
  for i := 0 to 255 do
     begin
        sEntry := IntToStr( pal.logPalette.palpalEntry[i].peRed ) + ',' +
           IntToStr( pal.logPalette.palpalEntry[i].peGreen ) + ',' +
           IntToStr( pal.logPalette.palpalEntry[i].peBlue ) + ',' +
           IntToStr( pal.logPalette.palpalEntry[i].peFlags );
        FEntries.Add( sEntry );
     end;
end;

procedure TColorPalette.SetBadEntry( str: TStrings );
begin
end;

function TColorPalette.GetPaletteEntry( n: byte ): TPaletteEntry;
begin
  Result := pal.logPalette.palpalEntry[n];
end;

procedure TColorPalette.SetPaletteEntry( n: byte; pal: TPaletteEntry );
var
  sEntry: string;
begin
  sEntry := IntToStr( pal.peRed ) + ',' +
     IntToStr( pal.peGreen ) + ',' +
     IntToStr( pal.peBlue ) + ',' +
     IntToStr( pal.peFlags );
  FEntries[n] := sEntry;
  self.pal.logPalette.palpalEntry[n] := pal;
  if not bAnimating then
    endAnimation( n, 1 );
end;

procedure TColorPalette.beginAnimation;
begin
  bAnimating := true;
end;

procedure TColorPalette.endAnimation( StartIndex, NbrOfEntries: integer );
var
  i: integer;
begin
  if FFlag <> pcReserved then
    raise Exception.Create( 'Palette Animation only valid if PaletteEntryFlag = pcReserved' );
  bAnimating := false;
  AnimatePalette( FPalette, StartIndex, NbrOfEntries, @self.pal.logPalette.palpalEntry[StartIndex] );
end;

procedure TColorPalette.Assign( Source: TPersistent );
begin
  if Source is TColorPalette then
     begin
        FEntries.Assign( ( Source as TColorPalette ).FEntries );
        PalEntryFlag := ( Source as TColorPalette ).PalEntryFlag;
        bEntries := FALSE;
        SetPaletteHandle( FALSE );
     end;
end;

procedure TColorPalette.Refresh;
begin
  bEntries := FALSE;
  SetPaletteHandle( FALSE );
end;

(*********************************************
Property editors.
*********************************************)
procedure TColorPaletteEditor.Edit;
var
  cpSelf: TColorPalette;
  oldFlag: TPalEntryFlag;
begin
  cpSelf := TColorPalette( GetComponent( 0 ) );
  with TfrmColorPaletteEditor.Create( nil ) do
     begin
        oldFlag := cpSelf.PalEntryFlag;
        cpSelf.PalEntryFlag := pcReserved;
        ColorPalette1.Assign( cpSelf );
        if ShowModal = mrOk then
          cpSelf.Assign( ColorPalette1 );
        cpSelf.PalEntryFlag := oldFlag;
        Release;
     end;
end;

function TColorPaletteEditor.GetValue: string;
begin
  Result := '(PaletteEntries)';
end;

function TColorPaletteEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

procedure Register;
begin
  RegisterPropertyEditor( TypeInfo( TStrings ), TColorPalette, 'PaletteEntries', TColorPaletteEditor );
  RegisterComponents( 'TurboSprite', [TColorPalette] );
end;

end.
